/// <reference path="MS3DPrinter-Intellisense.js" />

var pskNs = "http://schemas.microsoft.com/windows/2003/08/printing/printschemakeywords";
var psfNs = "http://schemas.microsoft.com/windows/2003/08/printing/printschemaframework";
var psk3dNs = "http://schemas.microsoft.com/3dmanufacturing/2013/01/pskeywords3d";
var ms3dprintNs = "http://schemas.microsoft.com/3dmanufacturing/2015/01/materials"
var ns0000Ns = "http://schemas.fabrikam.com/3MF/2013/01";
var xsiNs = "http://www.w3.org/2001/XMLSchema-instance";

//
// Update known keywords in a XMLNode to use the 3d namespace
//
function updateNodeTo3dNamespace(xmlNode) {

    var updateKeywords = new Array();

    updateKeywords[1] = "ns0000:Job3DToolCount";
    updateKeywords[2] = "ns0000:Job3DTools";
    updateKeywords[3] = "ns0000:Extruder1";
    updateKeywords[4] = "ns0000:Extruder2";
    updateKeywords[5] = "ns0000:Extruder3";
    updateKeywords[6] = "ns0000:Extruder4";
    updateKeywords[7] = "ns0000:Extruder5";
    updateKeywords[8] = "ns0000:MaterialTypeOption";
    updateKeywords[9] = "ns0000:MaterialColor";
    updateKeywords[10] = "ns0000:FilamentDiameterOption";
    updateKeywords[11] = "ns0000:Job3DInitialTool";
    updateKeywords[12] = "ns0000:Job3DOutputArea";
    updateKeywords[13] = "ns0000:Job3DOutputAreaWidth";
    updateKeywords[14] = "ns0000:Job3DOutputAreaDepth";
    updateKeywords[15] = "ns0000:Job3DOutputAreaHeight";
    updateKeywords[16] = "ns0000:Job3DToolTemperature";
    updateKeywords[17] = "ns0000:Job3DPlatformTemperature";
    updateKeywords[18] = "ns0000:Job3DDeviceLanguage";
    updateKeywords[29] = "ns0000:3MF";
    updateKeywords[20] = "ns0000:STL";
    updateKeywords[21] = "ns0000:AMF";
    updateKeywords[22] = "ns0000:GCode";
    updateKeywords[23] = "ns0000:Bitmap";
    updateKeywords[24] = "ns0000:Job3DMaterialType";
    updateKeywords[25] = "ns0000:PLA";
    updateKeywords[26] = "ns0000:ABS";
    updateKeywords[27] = "ns0000:HDPE";
    updateKeywords[28] = "ns0000:PVA";
    updateKeywords[29] = "ns0000:Composite";
    updateKeywords[30] = "ns0000:SolidWood";
    updateKeywords[31] = "ns0000:SolidMetal";
    updateKeywords[32] = "ns0000:Powder";
    updateKeywords[33] = "ns0000:Job3DFilamentFeedRate";
    updateKeywords[34] = "ns0000:Job3DFilamentDiameter";
    updateKeywords[35] = "ns0000:ExtraThin";
    updateKeywords[36] = "ns0000:Thin";
    updateKeywords[37] = "ns0000:Medium";
    updateKeywords[38] = "ns0000:Thick";
    updateKeywords[39] = "ns0000:Job3DFilamentNozzleDiameter";
    updateKeywords[40] = "ns0000:Job3DToolTravelRate";
    updateKeywords[41] = "ns0000:Job3DQuality";
    updateKeywords[42] = "ns0000:Draft";
    updateKeywords[43] = "ns0000:Medium";
    updateKeywords[44] = "ns0000:High";
    updateKeywords[45] = "ns0000:Job3DDensity";
    updateKeywords[46] = "ns0000:Hollow";
    updateKeywords[47] = "ns0000:Low";
    updateKeywords[48] = "ns0000:Medium";
    updateKeywords[49] = "ns0000:High";
    updateKeywords[50] = "ns0000:Solid";
    updateKeywords[51] = "ns0000:InfillPercentage";
    updateKeywords[52] = "ns0000:Job3DSliceHeight";
    updateKeywords[53] = "ns0000:Job3DShellCount";
    updateKeywords[54] = "ns0000:Job3DOutputColor";
    updateKeywords[55] = "ns0000:Color";
    updateKeywords[56] = "ns0000:Monochrome";
    updateKeywords[57] = "ns0000:Job3DSupports";
    updateKeywords[58] = "ns0000:SupportsIncluded";
    updateKeywords[59] = "ns0000:SupportsExcluded";
    updateKeywords[60] = "ns0000:Job3DRaft";
    updateKeywords[61] = "ns0000:RaftIncluded";
    updateKeywords[62] = "ns0000:RaftExcluded";
    updateKeywords[63] = "ns0000:InfillPercentage";
    updateKeywords[64] = "ns0000:Job3DKnownObjectTypes";
    updateKeywords[65] = "ns0000:MeshOnly";
    updateKeywords[66] = "ns0000:SlicesOnly";
    updateKeywords[67] = "ns0000:MeshAndSlices";



    try {
        if (xmlNode.nodeType == 1 || xmlNode.nodeType == 3) // Element or String only
        {
            // features
            var attrName = xmlNode.getAttribute("name");
            for (var index = 0; index < updateKeywords.length; index++) {
                if (xmlNode.nodeType == 1) {
                    // Check node name
                    if (attrName) {
                        if (attrName === updateKeywords[index]) {
                            xmlNode.setAttribute("name", "psk3d:" + attrName.substring(attrName.indexOf(":") + 1));
                            break;
                        }
                    }
                }
                else if (xmlNode.nodeType == 3) {
                    // Check node value
                    if (xmlNode === updateKeywords[index]) {
                        xmlNode = ("psk3d:" + xmlNode.substring(xmlNode.indexOf(":") + 1));
                        break;
                    }
                }
            }
        }
    }
    catch (e) {
        //        debug.innerHTML += (xmlNode.nodeName.toString() + ":" + e.toString() + "<br>");
    }
}

//
// Navigate DOM looking for things we need to change to the 3d namespace
//
function navigateAndUpdateDOM(xmlNode) {

    var Child;

    for (var ChildCount = 0; ChildCount < xmlNode.childNodes.length; ChildCount++) {
        Child = xmlNode.childNodes.item(ChildCount);
        updateNodeTo3dNamespace(Child);
        navigateAndUpdateDOM(Child);
    }
}

function addProperty(Root, pstrPropertyName, pstrNodeName, pstrValueType, pstrValue, pDocument) {
    var temp = createProperty(pstrPropertyName, pstrNodeName, pstrValueType, pstrValue, pDocument);
    Root.appendChild(temp);
}

function createProperty(pstrPropertyName, pstrNodeName, pstrValueType, pstrValue, pDocument) {
    var newNode = pDocument.XmlNode.createNode(1, pstrNodeName, psfNs);
    newNode.setAttribute("name", pstrPropertyName);

    if (pstrValueType.length > 0) {
        var newProp = pDocument.XmlNode.createNode(1, "psf:Value", psfNs);
        var newAttr = pDocument.XmlNode.createNode(2, "xsi:type", xsiNs);
        newAttr.nodeValue = pstrValueType;
        newProp.setAttributeNode(newAttr);

        var newText = pDocument.XmlNode.createTextNode(pstrValue);

        newProp.appendChild(newText);

        newNode.appendChild(newProp);
    }
    return newNode;
}

function completePrintCapabilities(printTicket, scriptContext, printCapabilities) {
    /// <param name="printTicket" type="IPrintSchemaTicket" mayBeNull="true">
    ///     If not 'null', the print ticket's settings are used to customize the print capabilities.
    /// </param>
    /// <param name="scriptContext" type="IPrinterScriptContext">
    ///     Script context object.
    /// </param>
    /// <param name="printCapabilities" type="IPrintSchemaCapabilities">
    ///     Print capabilities object to be customized.
    /// </param>

    var xmlCapabilities = printCapabilities.XmlNode;

    var Root;

    // find psf:PrintCapabilities
    for (var ChildCount = 0; ChildCount < xmlCapabilities.childNodes.length; ChildCount++) {
        Root = xmlCapabilities.childNodes.item(ChildCount);

        if (Root.nodeName == "psf:PrintCapabilities" || Root.nodeName == "psf:PrintTicket") {
            // Add new attribute xmlns:pVendorNode="http://schemas.microsoft.com/3dmanufacturing/2015/01/materials"
            Root.setAttribute("xmlns:ms3dprint", ms3dprintNs.toString());
            // Add new attribute xmlns:psk3d="http://schemas.microsoft.com/3dmanufacturing/2013/01/pskeywords3d""
            Root.setAttribute("xmlns:psk3d", psk3dNs.toString());
            //            debug.innerHTML +=  psk3dNs.toString() + "<br>";

            navigateAndUpdateDOM(Root);

            // Get device config
            var xmlDeviceConfig = scriptContext.QueueProperties.GetReadStreamAsXML("PrintDeviceCapabilities");

            // Parse config
            parseDeviceConfigFile(xmlDeviceConfig, PrintQualityType.PrintQualityType_Unknown);

            if (Root.nodeName == "psf:PrintCapabilities") {
                // Add default values
                // Job3DQuality
                var quality = createProperty("psk3d:Job3DQuality", "psf:Feature", "", "", printCapabilities);
                Root.appendChild(quality);
                var selectionTypeQuality = createProperty("psf:SelectionType", "psf:Property", "xsd:QName", "psk:PickOne", printCapabilities);
                quality.appendChild(selectionTypeQuality);
                var displayNameQuality = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Quality", printCapabilities);
                quality.appendChild(displayNameQuality);

                var highQuality = createProperty("psk3d:High", "psf:Option", "", "", printCapabilities);
                quality.appendChild(highQuality);
                var displayNameHigh = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "High", printCapabilities);
                highQuality.appendChild(displayNameHigh);
                highQuality.setAttribute("constrained", "psk:None");

                var mediumQuality = createProperty("psk3d:Medium", "psf:Option", "", "", printCapabilities);
                quality.appendChild(mediumQuality);
                var displayNameMedium = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Medium", printCapabilities);
                mediumQuality.appendChild(displayNameMedium);
                mediumQuality.setAttribute("constrained", "psk:None");

                var draftQuality = createProperty("psk3d:Draft", "psf:Option", "", "", printCapabilities);
                quality.appendChild(draftQuality);
                var displayNameDraft = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Draft", printCapabilities);
                draftQuality.appendChild(displayNameDraft);
                draftQuality.setAttribute("constrained", "psk:None");

                // Job3DDensity
                var density = createProperty("psk3d:Job3DDensity", "psf:Feature", "", "", printCapabilities);
                Root.appendChild(density);
                var selectionTypeDensity = createProperty("psf:SelectionType", "psf:Property", "xsd:QName", "psk:PickOne", printCapabilities);
                density.appendChild(selectionTypeDensity);
                var displayNameDensity = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Density", printCapabilities);
                density.appendChild(displayNameDensity);

                var hollowDensity = createProperty("psk3d:Hollow", "psf:Option", "", "", printCapabilities);
                density.appendChild(hollowDensity);
                var displayNameHollow = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Hollow", printCapabilities);
                hollowDensity.appendChild(displayNameHollow);
                hollowDensity.setAttribute("constrained", "psk:None");

                var lowDensity = createProperty("psk3d:Low", "psf:Option", "", "", printCapabilities);
                density.appendChild(lowDensity);
                var displayNameHollow = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Low", printCapabilities);
                lowDensity.appendChild(displayNameHollow);
                lowDensity.setAttribute("constrained", "psk:None");

                var mediumDensity = createProperty("psk3d:Medium", "psf:Option", "", "", printCapabilities);
                density.appendChild(mediumDensity);
                var displayNameMedium = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Medium", printCapabilities);
                mediumDensity.appendChild(displayNameMedium);
                mediumDensity.setAttribute("constrained", "psk:None");

                var highDensity = createProperty("psk3d:High", "psf:Option", "", "", printCapabilities);
                density.appendChild(highDensity);
                var displayNameHigh = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "High", printCapabilities);
                highDensity.appendChild(displayNameHigh);
                highDensity.setAttribute("constrained", "psk:None");

                var solidDensity = createProperty("psk3d:Solid", "psf:Option", "", "", printCapabilities);
                density.appendChild(solidDensity);
                var displayNameSolid = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Solid", printCapabilities);
                solidDensity.appendChild(displayNameSolid);
                solidDensity.setAttribute("constrained", "psk:None");

                // Job3DSupports
                var support = createProperty("psk3d:Job3DSupports", "psf:Feature", "", "", printCapabilities);
                Root.appendChild(support);
                var selectionTypeSupport = createProperty("psf:SelectionType", "psf:Property", "xsd:QName", "psk:PickOne", printCapabilities);
                support.appendChild(selectionTypeSupport);
                var displayNameSupport = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Include Object Support Structures", printCapabilities);
                support.appendChild(displayNameSupport);

                var includedSupport = createProperty("psk3d:SupportsIncluded", "psf:Option", "", "", printCapabilities);
                support.appendChild(includedSupport);
                var displayNameIncluded = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Yes", printCapabilities);
                includedSupport.appendChild(displayNameIncluded);
                includedSupport.setAttribute("constrained", "psk:None");

                var excludedSupport = createProperty("psk3d:SupportsExcluded", "psf:Option", "", "", printCapabilities);
                support.appendChild(excludedSupport);
                var displayNameExcluded = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "No", printCapabilities);
                excludedSupport.appendChild(displayNameExcluded);
                excludedSupport.setAttribute("constrained", "psk:None");

                // Job3DRaft
                var raft = createProperty("psk3d:Job3DRaft", "psf:Feature", "", "", printCapabilities);
                Root.appendChild(raft);
                var selectionTypeRaft = createProperty("psf:SelectionType", "psf:Property", "xsd:QName", "psk:PickOne", printCapabilities);
                raft.appendChild(selectionTypeRaft);
                var displayNameRaft = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Include Object Raft", printCapabilities);
                raft.appendChild(displayNameRaft);

                var includedRaft = createProperty("psk3d:RaftIncluded", "psf:Option", "", "", printCapabilities);
                raft.appendChild(includedRaft);
                var displayNameIncluded = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Yes", printCapabilities);
                includedRaft.appendChild(displayNameIncluded);
                includedRaft.setAttribute("constrained", "psk:None");

                var excludedRaft = createProperty("psk3d:RaftExcluded", "psf:Option", "", "", printCapabilities);
                raft.appendChild(excludedRaft);
                var displayNameExcluded = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "No", printCapabilities);
                excludedRaft.appendChild(displayNameExcluded);
                excludedRaft.setAttribute("constrained", "psk:None");

                // psk3d:AppName
                var appNameNode = createProperty("psk3d:AppName", "psf:Property", "xsd:string", params.appName, printCapabilities);
                Root.appendChild(appNameNode);

                // psk3d:CloudPrinter
                if (params.isCloudApp) {
                    var cloudAppNode = createProperty("psk3d:CloudPrinter", "psf:Property", "xsd:string", "true", printCapabilities);
                    Root.appendChild(cloudAppNode);
                }

                // psk3d:Job3DToolCount
                var toolCountNode = createProperty("psk3d:Job3DToolCount", "psf:Property", "xsd:integer", "1", printCapabilities);
                Root.appendChild(toolCountNode);

                // psk3d:Job3DTools
                var toolsNode = createProperty("psk3d:Job3DTools", "psf:Property", "", "", printCapabilities);
                Root.appendChild(toolsNode);
                var ext1Node = createProperty("psk3d:Extruder1", "psf:Property", "", "", printCapabilities);
                toolsNode.appendChild(ext1Node);
                var ext1NameNode = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Extruder", printCapabilities);
                ext1Node.appendChild(ext1NameNode);

                if (params.material_type_pla) {
                    var ext1MatType = createProperty("psk3d:MaterialTypeOption", "psf:Property", "xsd:QName", "psk3d:PLA", printCapabilities);
                    ext1Node.appendChild(ext1MatType);
                }

                if (params.material_type_abs) {
                    var ext1MatType = createProperty("psk3d:MaterialTypeOption", "psf:Property", "xsd:QName", "psk3d:ABS", printCapabilities);
                    ext1Node.appendChild(ext1MatType);
                }

                var ext1FilDia = createProperty("psk3d:FilamentDiameterOption", "psf:Property", "xsd:QName", "psk3d:Medium", printCapabilities);
                ext1Node.appendChild(ext1FilDia);

                // psk3d:JobOutputArea - in microns
                var outputAreaNode = createProperty("psk3d:Job3DOutputArea", "psf:Property", "", "", printCapabilities);
                Root.appendChild(outputAreaNode);
                var outWidth = createProperty("psk3d:Job3DOutputAreaWidth", "psf:Property", "xsd:integer", getValue(params.bedsizex), printCapabilities);
                outputAreaNode.appendChild(outWidth);
                var outDepth = createProperty("psk3d:Job3DOutputAreaDepth", "psf:Property", "xsd:integer", getValue(params.bedsizey), printCapabilities);
                outputAreaNode.appendChild(outDepth);

                var bedSizeZWithRaft = params.bedsizez;
                if (params.bRaft) {
                    bedSizeZWithRaft = params.bedsizez - params.raftLayers * params.sliceheight;
                }

                var outHeight = createProperty("psk3d:Job3DOutputAreaHeight", "psf:Property", "xsd:integer", getValue(bedSizeZWithRaft), printCapabilities);
                outputAreaNode.appendChild(outHeight);

                // psk3d:Job3DFilamentFeedRate - in microns/second
                var feedRateNode = createProperty("psk3d:Job3DFilamentFeedRate", "psf:Property", "xsd:integer", "30000", printCapabilities);
                Root.appendChild(feedRateNode);

                // psk3d:Job3DFilamentNozzleDiameter - diameter in microns 
                var nozzleDiaNode = createProperty("psk3d:Job3DFilamentNozzleDiameter", "psf:Property", "xsd:integer", getValue(params.extrusionWidth), printCapabilities);
                Root.appendChild(nozzleDiaNode);

                // psk3d:Job3DToolTravelRate - in microns/second
                var travelRateNode = createProperty("psk3d:Job3DToolTravelRate", "psf:Property", "xsd:integer", getValue(params.speed), printCapabilities);
                Root.appendChild(travelRateNode);

                // psk3d:Job3DKnownObjectTypes - supports MeshOnly, SlicesOnly, or MeshAndSlices 
                var knownObjectTypeNode = createProperty("psk3d:Job3DKnownObjectTypes", "psf:Property", "xsd:QName", "psk3d:MeshOnly", printCapabilities);
                Root.appendChild(knownObjectTypeNode);

                // psk3d:Job3DSliceHeight - ParameterDef
                var sliceHeightNode = createProperty("psk3d:Job3DSliceHeight", "psf:ParameterDef", "", "", printCapabilities);
                Root.appendChild(sliceHeightNode);
                var dataTypeNode = createProperty("psf:DataType", "psf:Property", "xsd:QName", "xsd:integer", printCapabilities);
                sliceHeightNode.appendChild(dataTypeNode);
                var defaultValNode = createProperty("psf:DefaultValue", "psf:Property", "xsd:integer", getValue(params.sliceheight), printCapabilities);
                sliceHeightNode.appendChild(defaultValNode);
                var maxValNode = createProperty("psf:MaxValue", "psf:Property", "xsd:integer", getValue(params.sliceMaxHeight), printCapabilities);
                sliceHeightNode.appendChild(maxValNode);
                var minValNode = createProperty("psf:MinValue", "psf:Property", "xsd:integer", getValue(params.sliceMinHeight), printCapabilities);
                sliceHeightNode.appendChild(minValNode);
                var multipleNode = createProperty("psf:Multiple", "psf:Property", "xsd:integer", "1", printCapabilities);
                sliceHeightNode.appendChild(multipleNode);
                var mandatoryNode = createProperty("psf:Mandatory", "psf:Property", "xsd:QName", "psk:Optional", printCapabilities);
                sliceHeightNode.appendChild(mandatoryNode);
                var unitTypeNode = createProperty("psf:UnitType", "psf:Property", "xsd:string", "microns", printCapabilities);
                sliceHeightNode.appendChild(unitTypeNode);

                // If there are enough materials provided, build the extruder mapping UI
                if (params.Materials.length >= params.nExtruders) {

                    for (var exidx = 0; exidx < params.nExtruders; exidx++) {

                        var materialType = createProperty("psk3d:Job3DExtruder" + (exidx+1), "psf:Feature", "", "", printCapabilities);
                        Root.appendChild(materialType);

                        var selectionTypeMaterialType = createProperty("psf:SelectionType", "psf:Property", "xsd:QName", "psk:PickOne", printCapabilities);
                        materialType.appendChild(selectionTypeMaterialType);

                        var displayNameMaterialType = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Extruder " + (exidx + 1) + " material", printCapabilities);
                        materialType.appendChild(displayNameMaterialType);

                        for (var matidx = 0; matidx < params.Materials.length; matidx++) {

                            var materialOption = createProperty("ms3dprint:Material" + matidx, "psf:Option", "", "", printCapabilities);
                            materialType.appendChild(materialOption);
                            var displayName = createProperty("psk:DisplayName", "psf:Property", "xsd:string", params.Materials[matidx].Name, printCapabilities);
                            materialOption.appendChild(displayName);
                            materialOption.setAttribute("constrained", "psk:None");

                        }
                    }
                }


                // if we have more than one extruders/materials, we can print raft and supports with those
                if ((params.nExtruders > 1) && (params.Materials.length > 1)) {
                    //
                    // psk3d:pJob3DSupportsMaterial - ParameterDef
                    //
                    {
                        var supportsMaterialNode = createProperty("psk3d:Job3DSupportsMaterial", "psf:ParameterDef", "", "", printCapabilities);
                        Root.appendChild(supportsMaterialNode);
                        var dataTypeNode = createProperty("psf:DataType", "psf:Property", "xsd:QName", "xsd:QName", printCapabilities);
                        supportsMaterialNode.appendChild(dataTypeNode);
                        var defaultValNode = createProperty("psf:DefaultValue", "psf:Property", "xsd:QName", "ms3dprint:MatB", printCapabilities);
                        supportsMaterialNode.appendChild(defaultValNode);
                        var maxLengthNode = createProperty("psf:MaxLength", "psf:Property", "xsd:integer", "1024", printCapabilities);
                        supportsMaterialNode.appendChild(maxLengthNode);
                        var minLengthNode = createProperty("psf:MinLength", "psf:Property", "xsd:integer", "1", printCapabilities);
                        supportsMaterialNode.appendChild(minLengthNode);
                        var mandatoryNode = createProperty("psf:Mandatory", "psf:Property", "xsd:QName", "psk:Optional", printCapabilities);
                        supportsMaterialNode.appendChild(mandatoryNode);
                        var unitTypeNode = createProperty("psf:UnitType", "psf:Property", "xsd:string", "characters", printCapabilities);
                        supportsMaterialNode.appendChild(unitTypeNode);
                    }
                    //
                    // psk3d:pJob3DRaftMaterial - ParameterDef
                    //
                    {
                        var raftMaterialNode = createProperty("psk3d:Job3DRaftMaterial", "psf:ParameterDef", "", "", printCapabilities);
                        Root.appendChild(raftMaterialNode);
                        var dataTypeNode = createProperty("psf:DataType", "psf:Property", "xsd:QName", "xsd:QName", printCapabilities);
                        raftMaterialNode.appendChild(dataTypeNode);
                        var defaultValNode = createProperty("psf:DefaultValue", "psf:Property", "xsd:QName", "ms3dprint:MatB", printCapabilities);
                        raftMaterialNode.appendChild(defaultValNode);
                        var maxLengthNode = createProperty("psf:MaxLength", "psf:Property", "xsd:integer", "1024", printCapabilities);
                        raftMaterialNode.appendChild(maxLengthNode);
                        var minLengthNode = createProperty("psf:MinLength", "psf:Property", "xsd:integer", "1", printCapabilities);
                        raftMaterialNode.appendChild(minLengthNode);
                        var mandatoryNode = createProperty("psf:Mandatory", "psf:Property", "xsd:QName", "psk:Optional", printCapabilities);
                        raftMaterialNode.appendChild(mandatoryNode);
                        var unitTypeNode = createProperty("psf:UnitType", "psf:Property", "xsd:string", "characters", printCapabilities);
                        raftMaterialNode.appendChild(unitTypeNode);
                    }
                }

                // if we have extruders and matching materials, send 'em up
                if (params.Materials.length >= params.nExtruders) {
                    var materialCountNode = createProperty("psk3d:Job3DMaterialCount", "psf:Property", "xsd:integer", params.nExtruders.toString(), printCapabilities);
                    Root.appendChild(materialCountNode);
                    var materialsNode = createProperty("psk3d:Job3DMaterials", "psf:Property", "", "", printCapabilities);
                    Root.appendChild(materialsNode);

                    for (var i = 1; i <= params.Materials.length; i++) {
                        var sxMap = "ms3dprint:Job3DMat" + String.fromCharCode('A'.charCodeAt() + i - 1) + "Map";
                        var xMapNode = createProperty(sxMap, "psf:ParameterDef", "", "", printCapabilities);
                        Root.appendChild(xMapNode);

                        var dataTypeNode = createProperty("psf:DataType", "psf:Property", "xsd:QName", "xsd:string", printCapabilities);
                        xMapNode.appendChild(dataTypeNode);
                        var minLengthNode = createProperty("psf:MinLength", "psf:Property", "xsd:integer", "1", printCapabilities);
                        xMapNode.appendChild(minLengthNode);
                        var maxLengthNode = createProperty("psf:MaxLength", "psf:Property", "xsd:integer", "1024", printCapabilities);
                        xMapNode.appendChild(maxLengthNode);
                        var mandatoryNode = createProperty("psf:Mandatory", "psf:Property", "xsd:QName", "psk:Optional", printCapabilities);
                        xMapNode.appendChild(mandatoryNode);
                        var unitTypeNode = createProperty("psf:UnitType", "psf:Property", "xsd:string", "characters", printCapabilities);
                        xMapNode.appendChild(unitTypeNode);
                        var smaterialSelected = "ms3dprint:Mat" + String.fromCharCode('A'.charCodeAt() + i - 1);
                        var materialSelectedNode = createProperty("psk3d:Job3DMaterialSelected", "psf:Property", "xsd:QName", smaterialSelected, printCapabilities);
                        xMapNode.appendChild(materialSelectedNode);
                        var matDefaultNode = createProperty("psf:DefaultValue", "psf:Property", "xsd:string", "", printCapabilities);
                        xMapNode.appendChild(matDefaultNode);

                        var sVndX = "ms3dprint:Mat" + String.fromCharCode('A'.charCodeAt() + i - 1);
                        var vndXNode = createProperty(sVndX, "psf:Property", "", "", printCapabilities);
                        materialsNode.appendChild(vndXNode);
                        var displayName = params.Materials[i - 1].Name

                        // material changes depending on advanced settings extruder naming, so display the extruder instead here
                        var displayNameNode = createProperty("psk:DisplayName", "psf:Property", "xsd:string", "Extruder " + i /*+ ": " + displayName*/, printCapabilities);
                        vndXNode.appendChild(displayNameNode);
                        var materialColor = params.Materials[i - 1].Color
                        var materialColorNode = createProperty("psk3d:MaterialColor", "psf:Property", "xsd:string", materialColor, printCapabilities);
                        vndXNode.appendChild(materialColorNode);
                    }
                }
            }
            else {
                //
                // Add all the static 3d Printing Properties to the PrintTicket not expressable in the GPD 
                //
                var sliceHeightNode = createProperty("psk3d:Job3DSliceHeight", "psf:ParameterInit", "xsd:integer", getValue(params.sliceheight), printCapabilities);
                Root.appendChild(sliceHeightNode);
            }
        }
    }
}

function convertPrintTicketToDevMode(printTicket, scriptContext, devModeProperties) {
    /// <param name="printTicket" type="IPrintSchemaTicket">
    ///     Print ticket to be converted to DevMode.
    /// </param>
    /// <param name="scriptContext" type="IPrinterScriptContext">
    ///     Script context object.
    /// </param>
    /// <param name="devModeProperties" type="IPrinterScriptablePropertyBag">
    ///     The DevMode property bag.
    /// </param>
}


function convertDevModeToPrintTicket(devModeProperties, scriptContext, printTicket) {
    /// <param name="devModeProperties" type="IPrinterScriptablePropertyBag">
    ///     The DevMode property bag.
    /// </param>
    /// <param name="scriptContext" type="IPrinterScriptContext">
    ///     Script context object.
    /// </param>
    /// <param name="printTicket" type="IPrintSchemaTicket">
    ///     Print ticket to be converted from the DevMode.
    /// </param>

}

function validatePrintTicket(printTicket, scriptContext) {
    /// <param name="printTicket" type="IPrintSchemaTicket">
    ///     Print ticket to be validated.
    /// </param>
    /// <param name="scriptContext" type="IPrinterScriptContext">
    ///     Script context object.
    /// </param>
    /// <returns type="Number" integer="true">
    ///     Integer value indicating validation status.
    ///         1 - Print ticket is valid and was not modified.
    ///         2 - Print ticket was modified to make it valid.
    ///         0 - Print ticket is invalid.
    /// </returns>

    return 1;
}

var PrintSchemaConstrainedSetting = {
    PrintSchemaConstrainedSetting_None: 0,
    PrintSchemaConstrainedSetting_PrintTicket: 1,
    PrintSchemaConstrainedSetting_Admin: 2,
    PrintSchemaConstrainedSetting_Device: 3
};

var STREAM_SEEK = {
    STREAM_SEEK_SET: 0,
    STREAM_SEEK_CUR: 1,
    STREAM_SEEK_END: 2
};

var PrintSchemaSelectionType = {
    PrintSchemaSelectionType_PickOne: 0,
    PrintSchemaSelectionType_PickMany: 1
};

//
// Parse and get parameters as follows:
//

function getValue(value) {
    return parseInt(Math.round(1000.0 * value + 0.1)).toString();
}

var params = {
    nExtruders: 1,
    Materials: [],
    sliceheight: 0.2,
    sliceMaxHeight: 1000.0,
    sliceMinHeight: 0.0001,
    extrusionWidth: 0.4,
    nShells: 3,
    shellOffset: 0.0,
    nTopLayers: 4,
    nBottomLayers: 4,
    fill: 0.1,
    fillhigh: 0.4,
    fillmedium: 0.2,
    filllow: 0.1,
    fillAngle: 45.0,
    filloverlap: 0.0,
    speed: 90.0,
    speedOuter: 50.0,
    speedFirst: 40.0,
    speedTravel: 120.0,
    speedRetract: 20.0,
    retraction: 1.0,
    bedsizex: 100.0,
    bedsizey: 100.0,
    bedsizez: 100.0,
    autocenter: 1,
    material_type_pla: false,
    material_type_abs: false,
    filament_diameter: 1.75,
    extruder1_temperature: 0.0,
    extruder2_temperature: 0.0,
    bExtruder1: false,
    bExtruder2: false,
    temperatureBed: 0.0,
    filament_calibration_override: 1.0,
    bRaft: false,
    raftLayers: 1,
    raftPathWidth: 2.0,
    raftFill: 0.2,
    raftZGap: 1.0,
    raftOffset: 8.0,
    raftSpeedFirst: 40.0,
    coolingtime: 5.0,
    minCoolingSpeed: 1.0,
    bridgingSpeed: 20.0,
    rampUpTarget: 0.0,
    appName: "",
    isCloudApp: false
};

var PrintQualityType = {
    PrintQualityType_Unknown: 0,
    PrintQualityType_High: 1,
    PrintQualityType_Medium: 2,
    PrintQualityType_Draft: 3
};

var PSK3DPrintTicket = {
    JOB3DUNKNOWN: 0,
    JOB3DQUALITY: 1,
    JOB3DDENSITY: 2,
    JOB3DSUPPORTS: 3,
    JOB3DMATERIALTYPE: 4,
    JOB3DRAFT: 5,
    JOB3DFILAMENTDIAMETER: 6,
    JOB3DOUTPUTCOLOR: 7,
    JOB3DSLICEHEIGHT: 8,
    JOB3DTOPSURFACELAYERS: 9,
    JOB3DBOTTOMSURFACELAYERS: 10
};

function parseDeviceConfigFile(xmlDeviceConfig, printQualityType) {
    var printChoice;
    if (printQualityType == PrintQualityType.PrintQualityType_Unknown) {
        if (xmlDeviceConfig.getElementsByTagName("psk3dx:high").length > 0 &&
            (xmlDeviceConfig.getElementsByTagName("psk3dx:high")[0].getAttribute("psk3dx:default") == "true" ||
            xmlDeviceConfig.getElementsByTagName("psk3dx:high")[0].getAttribute("default") == "true")) {
            printChoice = xmlDeviceConfig.getElementsByTagName("psk3dx:high")[0].childNodes;
        } else if (xmlDeviceConfig.getElementsByTagName("psk3dx:medium").length > 0 &&
            (xmlDeviceConfig.getElementsByTagName("psk3dx:medium")[0].getAttribute("psk3dx:default") == "true" ||
            xmlDeviceConfig.getElementsByTagName("psk3dx:medium")[0].getAttribute("default") == "true")) {
            printChoice = xmlDeviceConfig.getElementsByTagName("psk3dx:medium")[0].childNodes;
        } else if (xmlDeviceConfig.getElementsByTagName("psk3dx:draft").length > 0 &&
            (xmlDeviceConfig.getElementsByTagName("psk3dx:draft")[0].getAttribute("psk3dx:draft") == "true" ||
            xmlDeviceConfig.getElementsByTagName("psk3dx:draft")[0].getAttribute("default") == "true")) {
            printChoice = xmlDeviceConfig.getElementsByTagName("psk3dx:draft")[0].childNodes;
        }
    } else {
        if (printQualityType == PrintQualityType.PrintQualityType_High) {
            printChoice = xmlDeviceConfig.getElementsByTagName("psk3dx:high")[0].childNodes;
        } else if (printQualityType == PrintQualityType.PrintQualityType_Medium) {
            printChoice = xmlDeviceConfig.getElementsByTagName("psk3dx:medium")[0].childNodes;
        } else {
            printChoice = xmlDeviceConfig.getElementsByTagName("psk3dx:draft")[0].childNodes;
        }
    }

    params.bedsizex = parseFloat(xmlDeviceConfig.getElementsByTagName("psk3d:Job3DOutputAreaWidth")[0].text) / 1000;
    params.bedsizey = parseFloat(xmlDeviceConfig.getElementsByTagName("psk3d:Job3DOutputAreaDepth")[0].text) / 1000;
    params.bedsizez = parseFloat(xmlDeviceConfig.getElementsByTagName("psk3d:Job3DOutputAreaHeight")[0].text) / 1000;

    if (xmlDeviceConfig.getElementsByTagName("psk3d:autocenter").length > 0) {
        params.autocenter = parseInt(xmlDeviceConfig.getElementsByTagName("psk3d:autocenter")[0].text);
    }

    if (xmlDeviceConfig.getElementsByTagName("psk3d:pla").length > 0 || xmlDeviceConfig.getElementsByTagName("psk3d:GenericPLA").length > 0) {
        params.material_type_pla = true;
    }

    if (xmlDeviceConfig.getElementsByTagName("psk3d:abs").length > 0 || xmlDeviceConfig.getElementsByTagName("psk3d:GenericABS").length > 0) {
        params.material_type_abs = true;
    }

    params.filament_diameter = parseFloat(xmlDeviceConfig.getElementsByTagName("psk3dx:filamentdiameter")[0].text) / 1000;
    params.filament_calibration_override = parseFloat(xmlDeviceConfig.getElementsByTagName("psk3dx:filamentcalibrationoverride")[0].text);

    for (var i = 0; i < printChoice.length; i++) {
        var name = printChoice.item(i).nodeName
        if (name != null) {
            if (name == "psk3dx:layerthickness") {
                params.sliceheight = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:maxlayerthickness") {
                params.sliceMaxHeight = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:minlayerthickness") {
                params.sliceMinHeight = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:pathwidth") {
                params.extrusionWidth = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:shells") {
                params.nShells = parseInt(printChoice.item(i).text);
            } else if (name == "psk3dx:shelloffset") {
                params.shellOffset = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:topsurfacelayers") {
                params.nTopLayers = parseInt(printChoice.item(i).text);
            } else if (name == "psk3dx:bottomsurfacelayers") {
                params.nBottomLayers = parseInt(printChoice.item(i).text);
            } else if (name == "psk3dx:fill") {
                params.fill = parseFloat(printChoice.item(i).text);
            } else if (name == "psk3dx:filllow") {
                params.filllow = parseFloat(printChoice.item(i).text);
            } else if (name == "psk3dx:fillmedium") {
                params.fillmedium = parseFloat(printChoice.item(i).text);
            } else if (name == "psk3dx:fillhigh") {
                params.fillhigh = parseFloat(printChoice.item(i).text);
            } else if (name == "psk3dx:fillangle") {
                params.fillangle = parseFloat(printChoice.item(i).text);
            } else if (name == "psk3dx:filloverlap") {
                params.filloverlap = parseFloat(printChoice.item(i).text);
            } else if (name == "psk3dx:speed") {
                params.speed = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:speedouter") {
                params.speedouter = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:speedfirst") {
                params.speedfirst = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:speedtravel") {
                params.speedtravel = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:speedretract") {
                params.speedretract = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:retraction") {
                params.retraction = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:supportxygap") {
                params.supportxygap = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:supportfill") {
                params.supportfill = parseFloat(printChoice.item(i).text);
            } else if (name == "psk3dx:raftlayers") {
                params.raftlayers = parseInt(printChoice.item(i).text);
            } else if (name == "psk3dx:raftlayerthickness") {
                params.raftlayerthickness = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:raftpathwidth") {
                params.raftpathwidth = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:raftfill") {
                params.raftfill = parseFloat(printChoice.item(i).text);
            } else if (name == "psk3dx:raftzgap") {
                params.raftzgap = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:raftoffset") {
                params.raftoffset = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:raftspeedfirst") {
                params.raftspeedfirst = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:coolingtime") {
                params.coolingtime = parseFloat(printChoice.item(i).text);
            } else if (name == "psk3dx:mincoolingspeed") {
                params.mincoolingspeed = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:rampuptarget") {
                params.rampuptarget = parseFloat(printChoice.item(i).text) / 1000;
            } else if (name == "psk3dx:bridgingspeed") {
                params.bridgingspeed = parseFloat(printChoice.item(i).text) / 1000;
            }
        }
    }

    if (xmlDeviceConfig.getElementsByTagName("psk3dx:extrudertemperature").length > 0) {
        params.extruder1_temperature = parseFloat(xmlDeviceConfig.getElementsByTagName("psk3dx:extrudertemperature")[0].text);
        params.bExtruder1 = true;
    }

    if (xmlDeviceConfig.getElementsByTagName("psk3dx:extruder2temperature").length > 0) {
        params.extruder2_temperature = parseFloat(xmlDeviceConfig.getElementsByTagName("psk3dx:extruder2temperature")[0].text);
        params.bExtruder2 = true;
    }

    params.temperatureBed = parseFloat(xmlDeviceConfig.getElementsByTagName("psk3dx:platformtemperature")[0].text);

    if (xmlDeviceConfig.getElementsByTagName("psk3dx:extruders").length > 0) {
        params.nExtruders = parseFloat(xmlDeviceConfig.getElementsByTagName("psk3dx:extruders")[0].text);
        if (params.nExtruders > 0) {
            params.bExtruder1 = true;
        }
        if (params.nExtruders > 1) {
            params.bExtruder2 = true;
        }
    }


    var appName = xmlDeviceConfig.getElementsByTagName("psk3dx:AppName") ;
    for (var i = 0; i < appName.length; i++) {
        params.appName += appName[i].text;
    }

    if (xmlDeviceConfig.getElementsByTagName("psk3dx:CloudPrinter").length > 0) {
        if (xmlDeviceConfig.getElementsByTagName("psk3dx:CloudPrinter")[0].text == "true") {
            params.isCloudApp = true;
        }
    }

    var dispalyName = xmlDeviceConfig.getElementsByTagName("psk:DisplayName");
    var materialColor = xmlDeviceConfig.getElementsByTagName("psk3d:MaterialColor") ;
    for (var i = 0; i < dispalyName.length; i++) {
        var material = new Object();
        material.Name = dispalyName[i].text;
        material.Color = materialColor[i].text;
        params.Materials.push(material);
    }
}

function startPrintJob(usbJobContext, printerStream, printerBidiSchemaResponses) {
    ///<summary>
    ///     This method is invoked when the job starts printing.
    /// </summary>
    /// <param name="usbJobContext" type="IPrinterScriptUsbJobContext">
    ///    USB job context object.
    /// </param>
    /// <param name="printerStream" type="IPrinterScriptableSequentialStream">
    ///    Allows the script to Write/Read data from the attached USB device.
    /// </param>
    /// <param name="printerBidiSchemaResponses" type="IPrinterBidiSchemaResponses">
    ///    Object used by the script to store all status responses.
    /// </param>
    /// <returns type="IPrinterScriptUsbJobContextReturnCodes">
    ///     Returns one of the codes from the usbJobContext.ReturnCodes object.
    /// </returns>

    var retVal = usbJobContext.ReturnCodes.Success;
    var jobProperties = usbJobContext.jobPropertyBag;

    // Note: Attempting to retrieve job properties that don't exist will throw an exception.
    // If this failure is continuable, it should be caught and handled appropriately.
    try {
        var optionalValue = jobProperties.GetBool("Optional_Property");
    }
    catch (e)
    { /* Nothing to do. This 'Optional_Property' property isn't required to continue execution */ }

    // Determine if the device has a duplexing unit installed. This sample code assumes the device
    // cannot get into an error state.
    var duplexingUnitInstalled = deviceHelpers.isDuplexingUnitInstalled(printerStream);

    // Initialize the job properties. These are used from the writePrintData function
    jobProperties.SetBool("ManualDuplexJob", !duplexingUnitInstalled);
    jobProperties.SetBool("ManualDuplexBidiEventSent", false);
    jobProperties.SetInt32("PagesProcessed", 0);

    return retVal;
}

function writePrintData(usbJobContext, printData, printerStream, writePrintDataProgress, printerBidiSchemaResponses) {
    ///<summary>
    ///     This method is invoked when the data needs to be sent to the device
    /// </summary>
    /// <param name="usbJobContext" type="IPrinterScriptUsbJobContext">
    ///    USB job context object.
    /// </param>
    /// <param name="printData" type="Array">
    ///     Buffer containing data to be written to the device.
    /// </param>
    /// <param name="printerStream" type="IPrinterScriptableSequentialStream">
    ///     Allows the script to Write/Read data from the attached USB device.
    /// </param>
    /// <param name="writePrintDataProgress" type="IPrinterScriptUsbWritePrintDataProgress">
    ///     Allows the script to retrieve/update the count of bytes written to the device.
    /// </param>
    /// <param name="printerBidiSchemaResponses" type="IPrinterBidiSchemaResponses">
    ///    Object used by the script to store all status responses.
    /// </param>
    /// <returns type="IPrinterScriptUsbJobContextReturnCodes">
    ///     Returns one of the codes from the usbJobContext.ReturnCodes object.
    /// </returns>

    var retVal = usbJobContext.ReturnCodes.Success;
    var jobProperties = usbJobContext.jobPropertyBag;

    // This job property determines how many pages have been processed, to track odd/even pages
    // in a manually duplexed job. It does not reflect the number of pages sent to the device.
    var pagesProcessed = jobProperties.GetInt32("PagesProcessed");

    // Determines if the script needs to perform manual duplexing.
    var isManualDuplexJob = jobProperties.GetBool("ManualDuplexJob");

    var processedByteCount = 0;

    // Must be more than 2 bytes of data for a valid print stream
    if (printData.length < 2) {
        return usbJobContext.ReturnCodes.AbortTheJob;
    }

    // Find the page separator index in the current print data blob.
    var pageSeparatorIndex = deviceHelpers.findPageSeparatorIndex(printData, processedByteCount);

    // If there is no page separator marker in the input buffer, the length of the buffer is returned.
    var endOfPageFound = (pageSeparatorIndex < printData.length);

    // If a page marker was found before the end-of-data, increment the index by 2 so that
    // the page marker is also sent to the device
    if (endOfPageFound) {
        pageSeparatorIndex += 2;
    }

    // Slice one page of data from the input printData buffer.
    var pageData = printData.slice(processedByteCount, pageSeparatorIndex);

    // In a manually duplexed job, write even pages to the device, and odd pages to
    // a persistent stream for later processing during a call to endPrintJob.
    // Alternately, if this is an automatically duplexed job, write this data to the device directly.
    if ((pagesProcessed % 2 === 0) ||
        !isManualDuplexJob) {
        // Since this is an even page, write it to the device.
        processedByteCount = printerStream.Write(pageData);

        // If one entire page was written to the device, update the
        // 'IPrinterScriptUsbJobContext.PrintedPageCount' property. Else, this counter is
        // incremented in subsequent calls to the 'writePrintData' method.
        if (endOfPageFound &&
            processedByteCount === pageSeparatorIndex) {
            usbJobContext.PrintedPageCount++;
        }
    } else {
        // If this is an odd page, save the remaining bytes to a persistent stream
        // for later processing.
        processedByteCount =
            usbJobContext.TemporaryStreams[deviceHelpers.MANUAL_DUPLEX_STREAM].Write(pageData);
    }

    // If one page of data was found in the input buffer, and the entire page was
    // written, increment the processed page count. This counter is used to track odd/even
    // pages in a manually duplexed job.
    if (endOfPageFound &&
        processedByteCount === pageSeparatorIndex) {
        pagesProcessed++;
    }

    processedByteCount = pageSeparatorIndex;


    // Update the number of pages processed, for purpose of tracking odd/even pages.
    jobProperties.SetInt32("PagesProcessed", pagesProcessed);

    // Update 'IPrinterScriptUsbWritePrintDataProgress.ProcessedByteCount' property with the
    // number of bytes written, and USBMon will retry sending the rest of the buffer.
    writePrintDataProgress.ProcessedByteCount = processedByteCount;

    return retVal;
}

function endPrintJob(usbJobContext, printerStream, printerBidiSchemaResponses) {
    ///<summary>
    ///     This method is invoked when all the data to be written to the
    ///     device has been processed by the script.
    /// </summary>
    /// <param name="usbJobContext" type="IPrinterScriptUsbJobContext">
    ///    USB job context object.
    /// </param>
    /// <param name="printerStream" type="IPrinterScriptableSequentialStream">
    ///     Allows the script to Write/Read data from the attached USB device.
    /// </param>
    /// <param name="printerBidiSchemaResponses" type="IPrinterBidiSchemaResponses">
    ///    Object used by the script to store all status responses.
    /// </param>
    /// <returns type="IPrinterScriptUsbJobContextReturnCodes">
    ///     Returns one of the codes from the usbJobContext.ReturnCodes object.
    /// </returns>

    var retVal = usbJobContext.ReturnCodes.Success;
    var jobProperties = usbJobContext.jobPropertyBag;

    // Determines if the script needs to perform manual duplexing.
    var isManualDuplexJob = jobProperties.GetBool("ManualDuplexJob");

    // Is this a Manual duplex job (which implies pages stored in the 
    //  alternate persistent data stream
    if (isManualDuplexJob) {
        // Bubble up a Bidi event, that pops a toast, requesting the user to flip the pages.
        if (!jobProperties.GetBool("ManualDuplexBidiEventSent")) {
            printerBidiSchemaResponses.AddNull("\\Printer.Extension:ManualDuplexEvent");

            // Ensure the bidi event is not sent twice.
            jobProperties.SetBool("ManualDuplexBidiEventSent", true);
            retVal = usbJobContext.ReturnCodes.Retry;
        } else if (!deviceHelpers.isDeviceReadyForDuplexData(printerStream)) {

            // If the device hasn't been readied to receive duplex data, request USBMon to retry
            // this function after a short period of time.
            retVal = usbJobContext.ReturnCodes.Retry;
        } else {

            // If the device is ready to receive duplex data, try sending it one page of data.
            // If the device is unable to receive the entire page of data, store the unsent data
            // in a temporary stream and request USBMon to retry invoking this function.
            // On subsequent invocations, previously unwritten data from the temporary stream
            // will be written to the device before any new data from the manual duplex stream is
            // written out.

            retVal = usbJobContext.ReturnCodes.Retry;

            // Attempt to read data to previously unwritten data from the temporary stream. If
            // there is no such data, fresh data from the manual duplex stream is read.
            var streamsToRead = [deviceHelpers.UNWRITTEN_DATA_STREAM, deviceHelpers.MANUAL_DUPLEX_STREAM];
            var dataToWrite = null;

            for (var i = 0; i < streamsToRead.length; i++) {
                dataToWrite = usbJobContext.TemporaryStreams[streamsToRead[i]].Read(32768);

                if (dataToWrite &&
                    dataToWrite.length > 0) {
                    break;
                }
            }

            // If there is no data left to send in either stream, the job is complete.
            if (!dataToWrite ||
                dataToWrite.length === 0) {
                return usbJobContext.ReturnCodes.Success;
            }

            // Attempt to find a page separator index in the data.
            var pageSeparatorIndex = deviceHelpers.findPageSeparatorIndex(dataToWrite, 0);

            // If there is no page separator marker in the input buffer,
            // the length of the buffer is retured.
            var endOfPageFound = (pageSeparatorIndex < dataToWrite.length);

            // If a page marker was found before the end-of-data, increment the index by 2 so that
            // the page marker is also sent to the device
            if (endOfPageFound) {
                pageSeparatorIndex += 2;
            }

            var pageData = dataToWrite.slice(0, pageSeparatorIndex);

            // Try to write certain bytes to a device. If all of the data couldn't be written,
            // save it to a temporary stream.
            var bytesWritten = printerStream.Write(pageData);

            // If all the data in the buffer couldn't be written, either because the script only
            // sent data up to the page boundary, or because the device couldn't receive one page
            // of data, then store the data in a temporary buffer to be written to the device in
            // subsequent calls.
            if (bytesWritten < dataToWrite.length) {
                var unWrittenData = dataToWrite.slice(bytesWritten);

                var unwrittenSize =
                    usbJobContext.TemporaryStreams[deviceHelpers.UNWRITTEN_DATA_STREAM].Write(unWrittenData);

                // If the unwritten data couldn't be stored in the temporary stream,
                // this is an unrecoverable failure.
                if (unwrittenSize !== unWrittenData.length) {
                    retVal = usbJobContext.ReturnCodes.Failure;
                }
            }

            // If the entire buffer up to the page boundary was written,
            // increment the printed page count
            if (bytesWritten === pageSeparatorIndex &&
                endOfPageFound) {
                usbJobContext.PrintedPageCount++;
            }
        }
    }
    else {
        // Nothing to do for an automatically duplexed job.
    }

    return retVal;
}

var deviceHelpers = {
    isDuplexingUnitInstalled: function (printerStream) {
        /// <summary>
        ///     Query the device to determine if a duplexer unit
        ///     has been installed on the device.
        /// </summary>
        /// <param name="printerStream" type="IPrinterScriptableSequentialStream">
        ///    Allows the script to Write/Read data from the attached USB device.
        /// </param>
        /// <returns type="Boolean">
        ///     true - A duplexing unit has been installed on the device
        ///     false - A duplexing unit has not been installed on the device.
        ///             This script must perform manual duplexing
        /// </returns>
        var checkDuplexCommand = [0x0D, 0x0D, 0x02, 0xCA, 0xFE];

        // Ignore error handling because this check is device-specific.
        if (printerStream.Write(checkDuplexCommand)) {
            var response = printerStream.Read(1);
            if (response &&
                response[0] === 1) {
                return true;
            }
        }

        return false;
    },
    isDeviceReadyForDuplexData: function (printerStream) {
        /// <summary>
        ///     In a manual duplex job, queries the device to determine if the user has
        ///     flipped over printed pages and readied the device to receive manually
        ///     duplexed pages (such as by pressing a button).
        /// </summary>
        /// <param name="printerStream" type="IPrinterScriptableSequentialStream">
        ///    Allows the script to Write/Read data from the attached USB device.
        /// </param>
        /// <returns type="Boolean">
        ///     true - The user has flipped over the pages, and pressed a button on the device
        ///     false - The device is not ready to receive data yet
        /// </returns>
        var setDeviceManualDuplex = [0x58, 0x85];
        printerStream.Write(setDeviceManualDuplex);

        var deviceStatus = printerStream.Read(64);

        // Ignore error handling because this check is device-specific.
        if (deviceStatus &&
            deviceStatus.length >= 2 &&
            deviceStatus[0] === 0x1c && deviceStatus[1] === 0x02) {
            return true;
        } else {
            return false;
        }
    },
    findPageSeparatorIndex: function (buffer, startIndex) {
        /// <summary>
        ///     Retrieves the index in the input buffer at which the page separator
        ///     marker is located.
        /// </summary>
        /// <param name="buffer" type="Array">
        ///     Input data buffer
        /// </param>
        /// <param name="startIndex" type="Number" integer="true">
        ///     The index in the buffer from which the search begins
        /// </param>
        /// <returns type="Number" integer="true">
        ///     The index at which the PDL's page separator marker is located. Returns the
        ///     length of the input buffer if marker is not found in the buffer.
        /// </returns>

        // Note: The page separator marker used below has a two-byte signature of '0x5c, 0x72'
        var pageSeparatorMarker = [0x5c, 0x72];

        // Iterate through the input buffer to determine if there is a page separator marker.
        for (var i = startIndex; i < buffer.length - 1; i += pageSeparatorMarker.length) {
            var firstChar = buffer[i];
            var secondChar = buffer[i + 1];

            // Verify if the page separator marker is found at the current location.
            if (firstChar === pageSeparatorMarker[0]) {
                if (secondChar === pageSeparatorMarker[1]) {
                    return i;
                }
            }

            // If the second character being compared matches the first character of the
            // page separator maker, it is possible the second character could be the
            // beginning of a separator marker.
            if (secondChar === pageSeparatorMarker[0]) {
                i--;
            }
        }

        // No page separator marker was found in the input buffer.
        return buffer.length;
    },
    /// <summary>Represents the stream index where unwritten data is stored. </summary>
    UNWRITTEN_DATA_STREAM: 0,
    /// <summary>Represents the stream index where manually duplexed data it stored </summary>
    MANUAL_DUPLEX_STREAM: 1
};


